<?php
/**
 * Genesis Framework.
 *
 * WARNING: This file is part of the core Genesis Framework. DO NOT edit this file under any circumstances.
 * Please do all modifications in the form of a child theme.
 *
 * @package Genesis\Admin
 * @author  StudioPress
 * @license GPL-2.0+
 * @link    http://my.studiopress.com/themes/genesis/
 */

add_filter( 'user_contactmethods', 'genesis_user_contactmethods' );
/**
 * Filter the contact methods registered for users.
 *
 * Currently just adds a Google+ field.
 *
 * @since 1.9.0
 *
 * @param array $contactmethods Array of contact methods.
 */
function genesis_user_contactmethods( array $contactmethods ) {

	$contactmethods['googleplus'] = __( 'Google+', 'genesis' );

	return $contactmethods;

}

add_action( 'show_user_profile', 'genesis_user_options_fields' );
add_action( 'edit_user_profile', 'genesis_user_options_fields' );
/**
 * Add fields for user permissions for Genesis features to the user edit screen.
 *
 * Checkbox settings are:
 *
 * * Enable Genesis Admin Menu?
 * * Enable SEO Settings Submenu?
 * * Enable Import/Export Submenu?
 *
 * @since 1.4.0
 *
 * @param \WP_User $user User object.
 *
 * @return false Return false if current user can not edit users.
 */
function genesis_user_options_fields( $user ) {

	if ( ! current_user_can( 'edit_users', $user->ID ) )
		return false;

	?>
	<h3><?php _e( 'User Permissions', 'genesis' ); ?></h3>
	<table class="form-table">
		<tbody>
			<tr>
				<th scope="row"><?php _e( 'Genesis Admin Menus', 'genesis' ); ?></th>
				<td>
					<label for="genesis-meta[genesis_admin_menu]"><input id="genesis-meta[genesis_admin_menu]" name="genesis-meta[genesis_admin_menu]" type="checkbox" value="1" <?php checked( get_the_author_meta( 'genesis_admin_menu', $user->ID ) ); ?> />
					<?php _e( 'Enable Genesis Admin Menu?', 'genesis' ); ?></label><br />

					<?php if ( ! genesis_seo_disabled() ) : ?>
					<label for="genesis-meta[genesis_seo_settings_menu]"><input id="genesis-meta[genesis_seo_settings_menu]" name="genesis-meta[genesis_seo_settings_menu]" type="checkbox" value="1" <?php checked( get_the_author_meta( 'genesis_seo_settings_menu', $user->ID ) ); ?> />
					<?php _e( 'Enable SEO Settings Submenu?', 'genesis' ); ?></label><br />
					<?php endif; ?>

					<label for="genesis-meta[genesis_import_export_menu]"><input id="genesis-meta[genesis_import_export_menu]" name="genesis-meta[genesis_import_export_menu]" type="checkbox" value="1" <?php checked( get_the_author_meta( 'genesis_import_export_menu', $user->ID ) ); ?> />
					<?php _e( 'Enable Import/Export Submenu?', 'genesis' ); ?></label>
				</td>
			</tr>
		</tbody>
	</table>
	<?php

}

add_action( 'show_user_profile', 'genesis_user_archive_fields' );
add_action( 'edit_user_profile', 'genesis_user_archive_fields' );
/**
 * Add fields for author archives contents to the user edit screen.
 *
 * Input / Textarea fields are:
 *
 * * Custom Archive Headline
 * * Custom Description Text
 *
 * Checkbox fields are:
 *
 * * Enable Author Box on this User's Posts?
 * * Enable Author Box on this User's Archives?
 *
 * @since 1.6.0
 *
 * @param \WP_User $user User object.
 *
 * @return false Return false if current user can not edit users.
 */
function genesis_user_archive_fields( $user ) {

	if ( ! current_user_can( 'edit_users', $user->ID ) )
		return false;

	?>
	<h3><?php _e( 'Author Archive Settings', 'genesis' ); ?></h3>
	<p><span class="description"><?php _e( 'These settings apply to this author\'s archive pages.', 'genesis' ); ?></span></p>
	<table class="form-table">
		<tbody>
			<tr>
				<th scope="row"><label for="headline"><?php _e( 'Custom Archive Headline', 'genesis' ); ?></label></th>
				<td>
					<input name="genesis-meta[headline]" id="genesis-meta[headline]" type="text" value="<?php echo esc_attr( get_the_author_meta( 'headline', $user->ID ) ); ?>" class="regular-text" /><br />
					<span class="description"><?php printf( __( 'Will display in the %s tag at the top of the first page', 'genesis' ), genesis_code( '<h1>' ) ); ?></span>
				</td>
			</tr>

			<tr>
				<th scope="row"><label for="intro_text"><?php _e( 'Custom Description Text', 'genesis' ); ?></label></th>
				<td>
					<textarea name="genesis-meta[intro_text]" id="intro_text" rows="5" cols="30"><?php echo esc_textarea( get_the_author_meta( 'intro_text', $user->ID ) ); ?></textarea><br />
					<span class="description"><?php _e( 'This text will be the first paragraph, and display on the first page', 'genesis' ); ?></span>
				</td>
			</tr>

			<tr>
				<th scope="row"><?php _e( 'Author Box', 'genesis' ); ?></th>
				<td>
					<label for="genesis-meta[genesis_author_box_single]"><input id="genesis-meta[genesis_author_box_single]" name="genesis-meta[genesis_author_box_single]" type="checkbox" value="1" <?php checked( get_the_author_meta( 'genesis_author_box_single', $user->ID ) ); ?> />
					<?php _e( 'Enable Author Box on this User\'s Posts?', 'genesis' ); ?></label><br />

					<label for="genesis-meta[genesis_author_box_archive]"><input id="genesis-meta[genesis_author_box_archive]" name="genesis-meta[genesis_author_box_archive]" type="checkbox" value="1" <?php checked( get_the_author_meta( 'genesis_author_box_archive', $user->ID ) ); ?> />
					<?php _e( 'Enable Author Box on this User\'s Archives?', 'genesis' ); ?></label>
				</td>
			</tr>
		</tbody>
	</table>
	<?php

}

add_action( 'show_user_profile', 'genesis_user_seo_fields' );
add_action( 'edit_user_profile', 'genesis_user_seo_fields' );
/**
 * Add fields for author archive SEO to the user edit screen.
 *
 * Input / Textarea fields are:
 *
 * * Custom Document Title
 * * Meta Description
 * * Meta Keywords
 *
 * Checkbox fields are:
 *
 * * Apply noindex to this archive?
 * * Apply nofollow to this archive?
 * * Apply noarchive to this archive?
 *
 * @since 1.4.0
 *
 * @param \WP_User $user User object.
 *
 * @return false Return false if current user can not edit users.
 */
function genesis_user_seo_fields( $user ) {

	if ( ! current_user_can( 'edit_users', $user->ID ) )
		return false;

	?>
	<h3><?php _e( 'Author Archive SEO Settings', 'genesis' ); ?></h3>
	<p><span class="description"><?php _e( 'These settings apply to this author\'s archive pages.', 'genesis' ); ?></span></p>
	<table class="form-table">
		<tbody>
			<tr>
				<th scope="row"><label for="doctitle"><?php _e( 'Custom Document Title', 'genesis' ); ?></label></th>
				<td>
					<input name="genesis-meta[doctitle]" id="genesis-meta[doctitle]" type="text" value="<?php echo esc_attr( get_the_author_meta( 'doctitle', $user->ID ) ); ?>" class="regular-text" />
				</td>
			</tr>

			<tr>
				<th scope="row"><label for="meta-description"><?php _e( 'Meta Description', 'genesis' ); ?></label></th>
				<td>
					<textarea name="genesis-meta[meta_description]" id="genesis-meta[meta-description]" rows="5" cols="30"><?php echo esc_textarea( get_the_author_meta( 'meta_description', $user->ID ) ); ?></textarea>
				</td>
			</tr>

			<tr>
				<th scope="row"><label for="meta-keywords"><?php _e( 'Meta Keywords', 'genesis' ); ?></label></th>
				<td>
					<input name="genesis-meta[meta_keywords]" id="genesis-meta[meta-keywords]" type="text" value="<?php echo esc_attr( get_the_author_meta( 'meta_keywords', $user->ID ) ); ?>" class="regular-text" /><br />
					<span class="description"><?php _e( 'Comma separated list', 'genesis' ); ?></span>
				</td>
			</tr>

			<tr>
				<th scope="row"><?php _e( 'Robots Meta', 'genesis' ); ?></th>
				<td>
					<label for="genesis-meta[noindex]"><input id="genesis-meta[noindex]" name="genesis-meta[noindex]" type="checkbox" value="1" <?php checked( get_the_author_meta( 'noindex', $user->ID ) ); ?> />
					<?php printf( __( 'Apply %s to this archive?', 'genesis' ), genesis_code( 'noindex' ) ); ?></label><br />

					<label for="genesis-meta[nofollow]"><input id="genesis-meta[nofollow]" name="genesis-meta[nofollow]" type="checkbox" value="1" <?php checked( get_the_author_meta( 'nofollow', $user->ID ) ); ?> />
					<?php printf( __( 'Apply %s to this archive?', 'genesis' ), genesis_code( 'nofollow' ) ); ?></label><br />

					<label for="genesis-meta[noarchive]"><input id="genesis-meta[noarchive]" name="genesis-meta[noarchive]" type="checkbox" value="1" <?php checked( get_the_author_meta( 'noarchive', $user->ID ) ); ?> />
					<?php printf( __( 'Apply %s to this archive?', 'genesis' ), genesis_code( 'noarchive' ) ); ?></label>
				</td>
			</tr>
		</tbody>
	</table>
	<?php

}

add_action( 'show_user_profile', 'genesis_user_layout_fields' );
add_action( 'edit_user_profile', 'genesis_user_layout_fields' );
/**
 * Add author archive layout selector to the user edit screen.
 *
 * @since 1.4.0
 *
 * @uses genesis_layout_selector() Layout selector.
 *
 * @param \WP_User $user User object.
 *
 * @return false Return false if current user can not edit users.
 */
function genesis_user_layout_fields( $user ) {

	if ( ! current_theme_supports( 'genesis-archive-layouts' ) ) {
		return;
	}

	if ( ! current_user_can( 'edit_users', $user->ID ) )
		return false;

	$layout = get_the_author_meta( 'layout', $user->ID );
	$layout = $layout ? $layout : '';

	?>
	<h3><?php _e( 'Layout Settings', 'genesis' ); ?></h3>
	<p><span class="description"><?php _e( 'These settings apply to this author\'s archive pages.', 'genesis' ); ?></span></p>
	<table class="form-table">
		<tbody>
			<tr>
				<th scope="row"><?php _e( 'Choose Layout', 'genesis' ); ?></th>
				<td>
					<fieldset class="genesis-layout-selector">
						<legend class="screen-reader-text"><?php _e( 'Choose Layout', 'genesis' ); ?></legend>

						<p><input type="radio" name="genesis-meta[layout]" class="default-layout" id="default-layout" value="" <?php checked( $layout, '' ); ?> /> <label class="default" for="default-layout"><?php printf( __( 'Default Layout set in <a href="%s">Theme Settings</a>', 'genesis' ), menu_page_url( 'genesis', 0 ) ); ?></label></p>
						<?php genesis_layout_selector( array( 'name' => 'genesis-meta[layout]', 'selected' => $layout, 'type' => 'site' ) ); ?>

					</fieldset>
				</td>
			</tr>
		</tbody>
	</table>
	<?php

}

add_action( 'personal_options_update',  'genesis_user_meta_save' );
add_action( 'edit_user_profile_update', 'genesis_user_meta_save' );
/**
 * Update user meta when user edit page is saved.
 *
 * @since 1.4.0
 *
 * @param integer $user_id User ID
 *
 * @return null Returns null if current user can not edit users, or no meta fields submitted.
 */
function genesis_user_meta_save( $user_id ) {

	if ( ! current_user_can( 'edit_users', $user_id ) )
		return;

	if ( ! isset( $_POST['genesis-meta'] ) || ! is_array( $_POST['genesis-meta'] ) )
		return;

	$defaults = array(
		'genesis_admin_menu'         => '',
		'genesis_seo_settings_menu'  => '',
		'genesis_import_export_menu' => '',
		'genesis_author_box_single'  => '',
		'genesis_author_box_archive' => '',
		'headline'                   => '',
		'intro_text'                 => '',
		'doctitle'                   => '',
		'meta_description'           => '',
		'meta_keywords'              => '',
		'noindex'                    => '',
		'nofollow'                   => '',
		'noarchive'                  => '',
		'layout'                     => '',
	);

	/**
	 * Filter the user meta defaults array.
	 *
	 * Allows developer to filter the default array of user meta key => value pairs.
	 *
	 * @since 2.1.0
	 *
	 * @param array $defaults Default user meta array.
	 */
	$defaults = apply_filters( 'genesis_user_meta_defaults', $defaults );

	$meta = wp_parse_args( $_POST['genesis-meta'], $defaults );

	$meta['headline']   = strip_tags( $meta['headline'] );
	$meta['intro_text'] = current_user_can( 'unfiltered_html' ) ? $meta['intro_text'] : genesis_formatting_kses( $meta['intro_text'] );

	foreach ( $meta as $key => $value )
		update_user_meta( $user_id, $key, $value );

}

add_filter( 'get_the_author_genesis_admin_menu',         'genesis_user_meta_default_on', 10, 2 );
add_filter( 'get_the_author_genesis_seo_settings_menu',  'genesis_user_meta_default_on', 10, 2 );
add_filter( 'get_the_author_genesis_import_export_menu', 'genesis_user_meta_default_on', 10, 2 );
/**
 * Check to see if user data has actually been saved, or if defaults need to be forced.
 *
 * This filter is useful for user options that need to be "on" by default, but keeps us from having to push defaults
 * into the database, which would be a very expensive task.
 *
 * @since 1.4.0
 *
 * @global bool|object authordata User object if successful, false if not.
 *
 * @param string|boolean $value   The submitted value.
 * @param integer        $user_id User ID.
 *
 * @return string|integer Submitted value, or 1.
 */
function genesis_user_meta_default_on( $value, $user_id ) {

	//* Get the name of the field by removing the prefix from the active filter
	$field = str_replace( 'get_the_author_', '', current_filter() );

	//* If a real value exists, simply return it
	if ( $value )
		return $value;

	//* Setup user data
	if ( ! $user_id )
		global $authordata;
	else
		$authordata = get_userdata( $user_id );

	//* Just in case
	$user_field = "user_$field";
	if ( isset( $authordata->$user_field ) )
		return $authordata->user_field;

	//* If an empty or false value exists, return it
	if ( isset( $authordata->$field ) )
		return $value;

	//* If all that fails, default to true
	return 1;

}

add_filter( 'get_the_author_genesis_author_box_single', 'genesis_author_box_single_default_on', 10, 2 );
/**
 * Conditionally force a default 1 value for each users' author box setting.
 *
 * @since 1.4.0
 *
 * @uses genesis_get_option()           Get Genesis setting.
 * @uses genesis_user_meta_default_on() Get enforced conditional.
 *
 * @param string  $value   Submitted value.
 * @param integer $user_id User ID.
 *
 * @return string Result.
 */
function genesis_author_box_single_default_on( $value, $user_id ) {

	if ( genesis_get_option( 'author_box_single' ) )
		return genesis_user_meta_default_on( $value, $user_id );
	else
		return $value;

}
